/**
 * \file: helper.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: Key Handling Daemon
 *
 * \author: Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2016 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>

#include "helper.h"
#include "daemon_log.h"

#define SDC_DIR_PERM (S_IRWXU | S_IRGRP | S_IXGRP)

static sdc_error_t sdc_helper_create_dir_internal (const char *path)
{
    /* exit if directory does already exist */
    if (!access(path, F_OK))
        return SDC_OK;

    if (errno != ENOENT) {
        /* We never should get anything other than ENOENT */
        daemon_log(DAEMON_LOG_ERROR, "Unexpected error when checking access to %s - error: %s\n", path, strerror(errno));
        return SDC_INTERNAL_ERROR;
    }

    if (mkdir(path, SDC_DIR_PERM)) {
        /* we might get an error in case we have no write permissions for the parent folder - but this is unexpected*/
        daemon_log(DAEMON_LOG_ERROR, "mkdir failed for %s - error: %s\n", path, strerror(errno));
        return SDC_DAEMON_MKDIR_FAILED;
    }
    daemon_log(DAEMON_LOG_INFO, "Directory %s created\n", path);

    return SDC_OK;
}

static sdc_error_t sdc_helper_check_dir_internal (const char *path,
                                                  uid_t sdc_uid, gid_t sdc_gid,
                                                  bool last, bool warn_parent_permissions)
{
    struct stat stat_buf;
    sdc_error_t err = SDC_OK;

    if (stat(path, &stat_buf)) {
        daemon_log(DAEMON_LOG_ERROR, "Stat failed for %s - error: %s\n", path, strerror(errno));
        return SDC_UNKNOWN_ERROR;
    }

    if (!S_ISDIR(stat_buf.st_mode)) {
        /* Unexpected error */
        daemon_log(DAEMON_LOG_ERROR, "Path %s is no directory\n", path);
        err = SDC_UNKNOWN_ERROR;
    }

    if (last) {
        /* user/group and mode need to match exactly for the final part of the path */
        if (stat_buf.st_uid != sdc_uid) {
            daemon_log(DAEMON_LOG_ERROR, "Path %s is not owned by sdc uid\n", path);
            err = SDC_INTERNAL_ERROR;
        }
        if (stat_buf.st_gid != sdc_gid) {
            daemon_log(DAEMON_LOG_ERROR, "Path %s is not owned by sdc gid\n", path);
            err = SDC_INTERNAL_ERROR;
        }
        if ((stat_buf.st_mode & ALLPERMS) != SDC_DIR_PERM) {
            daemon_log(DAEMON_LOG_ERROR, "Mode of path %s does not match expectation 0x%x != 0x%x\n",
                       path,
                       (stat_buf.st_mode & ALLPERMS),
                       SDC_DIR_PERM);
            err = SDC_INTERNAL_ERROR;
        }
    } else {
        if (warn_parent_permissions && (!(stat_buf.st_mode & __S_ISVTX))) {
            /*
             * without sticky bit we need to print a warning in case the uid
             * or gid is different from root or sdc_uid/sdc_gid and
             * write permissions are set
             */
            if ( (stat_buf.st_mode & S_IWUSR) && (stat_buf.st_uid != sdc_uid) && (stat_buf.st_uid != 0) )
                daemon_log(DAEMON_LOG_WARNING, "WARNING: Uid of path %s might delete any sub-folder\n", path);

            if ( (stat_buf.st_mode & S_IWGRP) && (stat_buf.st_gid != sdc_gid) && (stat_buf.st_gid != 0) )
                daemon_log(DAEMON_LOG_WARNING, "WARNING: Gid of path %s might delete any sub-folder\n", path);

            /* others should never have any write permissions on these folders */
            if (stat_buf.st_mode & S_IWOTH)
                daemon_log(DAEMON_LOG_WARNING, "WARNING: Others might delete any sub-folder of path %s\n", path);
        }
    }

    return err;
}


sdc_error_t sdc_helper_create_absolute_dir(const char *abs_path,
                                           bool create_parent,
                                           bool warn_parent_permissions)
{
    sdc_error_t err = SDC_OK;
    char *dup_path;
    char *next; /* point to the next start of the search */
    char *curr; /* used to mark the current occurrence of / found */
    bool last; /* is last */
    mode_t oldmask;
    char *check_path;
    uid_t sdc_uid;
    gid_t sdc_gid;

    /* early abort if path exists */
    if (!access(abs_path, F_OK)) {
        /* path exists */
        return SDC_OK;
    }
    if ((errno != ENOENT)) {
        /* ignore - this will be handled when trying to access the folder */
        return SDC_OK;
    }

    dup_path = strdup(abs_path);
    if (!dup_path) {
        daemon_log(DAEMON_LOG_ERROR, "sdc_helper_create_dir failed to allocate memory\n");
        return SDC_NO_MEM;
    }

    /* store umask and set umask to 0*/
    oldmask = umask(0);

    /* get effective uid and gid of daemon */
    sdc_uid = getuid();
    sdc_gid = getgid();

    next = dup_path;
    last = false;

    while ((!last) && (err == SDC_OK)) {
        curr = strchr(next, '/');

        if (!curr) {
            /* we reached the last part */
            last = true;
        } else {
            /* replace '/' by '\0' */
            next = curr + 1;
            *curr = '\0';
        }

        /* required to handle root dir "/" (i.e. dup_path="") */
        check_path = "/";

        if (*dup_path !='\0') {
            check_path = dup_path;

            if (create_parent || last)
                err = sdc_helper_create_dir_internal(dup_path);
        }

        if (err == SDC_OK) {
            err = sdc_helper_check_dir_internal(check_path,
                                                sdc_uid, sdc_gid,
                                                last, warn_parent_permissions);
        }

        /* restore '/' */
        if (curr) {
            *curr = '/';
        }
    }

    /* recover umask */
    umask(oldmask);

    free(dup_path);

    return err;
}
